using System;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using SUPPLY = gov.va.med.vbecs.Common.VbecsTables.Supply;
using SUPPLYTYPE = gov.va.med.vbecs.Common.VbecsTables.SupplyType;
namespace gov.va.med.vbecs.BOL
{

	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Cameron Taylor</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>1/14/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>Supply business layer class</summary>

	#endregion
	
	
	public class Supply : BaseBusinessObject
	{

		#region Variables

		/// <summary>
		/// Data for our supply record
		/// </summary>
		private System.Data.DataRow _drSupplyRow;
		
		/// <summary>
		/// DataTable of our supply record
		/// </summary>
		private System.Data.DataTable _dtSupply;


		/// <summary>
		/// Exception that may have been generated because supply was selected with expired date
		/// UC24-26
		/// </summary>
		private BOL.ExceptionReport _lotException = null;

		#endregion

		#region Constructors

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="23"> 
		///		<ExpectedInput>Guid</ExpectedInput>
		///		<ExpectedOutput>Supply Object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="24"> 
		///		<ExpectedInput>New Guid</ExpectedInput>
		///		<ExpectedOutput>New Supply Object</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Supply class constructor
		/// </summary>
		public Supply(System.Guid supplyGUID)
		{

			this._dtSupply = DAL.Supply.GetSupplyByGuid(supplyGUID);

			if (this._dtSupply.Rows.Count == 0)
			{
				this._drSupplyRow = this._dtSupply.NewRow();
				this._drSupplyRow[SUPPLY.SupplyGuid] = supplyGUID;
				this.IsNew = true;
				this.IsDirty = true;
			}
			else
			{
				this.LoadFromDataRow(this._dtSupply.Rows[0]);
				this.IsNew = false;
				this.IsDirty = false;
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="25"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>Supply Object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="26"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates Supply object and populates it with data from DataRow
		/// </summary>
		/// <param name="dr"></param>
		public Supply(System.Data.DataRow dr)
		{
			this._dtSupply = DAL.Supply.GetEmptySupplyTable();

			this.LoadFromDataRow(dr);
			this.IsNew = false;
			this.IsDirty = false;
		}
		

		#endregion

		#region Properties

		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4389"> 
		///		<ExpectedInput>int</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4390"> 
		///		<ExpectedInput>negative number</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// SupplyName
		/// </summary>
		/// 
		public int AvailableQuantity
		{
			get
			{
				if (this._drSupplyRow.IsNull(SUPPLY.AvailableQuantity))
					return 0;

				return System.Convert.ToInt32(this._drSupplyRow[SUPPLY.AvailableQuantity]);
			}
			set
			{
				if (value < 0 || value > this.LotQuantity)
				{
					RuleBroken(MethodBase.GetCurrentMethod().Name, true);
					return;
				}

				RuleBroken(MethodBase.GetCurrentMethod().Name, false);
				this._drSupplyRow[SUPPLY.AvailableQuantity] = value;
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="27"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="28"> 
		///		<ExpectedInput>empty String</ExpectedInput>
		///		<ExpectedOutput>broken rule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// SupplyName
		/// </summary>
		public string SupplyName
		{
			get
			{
				if (this._drSupplyRow.IsNull(SUPPLY.SupplyName))
		
					return string.Empty;

				return this._drSupplyRow[SUPPLY.SupplyName].ToString();
			}
			set
			{
				if (value == string.Empty || value.Length > 75)
				{
					RuleBroken(MethodBase.GetCurrentMethod().Name, true);
					return;
				}

				RuleBroken(MethodBase.GetCurrentMethod().Name, false);
				this._drSupplyRow[SUPPLY.SupplyName] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="29"> 
		///		<ExpectedInput>DateTime</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="30"> 
		///		<ExpectedInput>Future Date</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ReceiptDate
		/// </summary>
		public System.DateTime ReceiptDate
		{
			get
			{
				if (this._drSupplyRow.IsNull(SUPPLY.ReceiptDate))
					return System.DateTime.MinValue;

				return (System.DateTime)this._drSupplyRow[SUPPLY.ReceiptDate];
			}
			set
			{
				if (value == System.DateTime.MinValue || value > BOL.VBECSDateTime.GetDivisionCurrentDateTime())
				{
					RuleBroken(MethodBase.GetCurrentMethod().Name, true);
					return;
				}

				//We'll break the LotExpirationDate, since it seems like it is the one that should change
				if (value > this.LotExpirationDate)
				{
					RuleBroken(Common.VbecsBrokenRules.FrmLogInSupplies.LotExpirationDate, true);
				}

				RuleBroken(MethodBase.GetCurrentMethod().Name, false);
				this._drSupplyRow[SUPPLY.ReceiptDate] = value;
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="31"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="32"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ReceiptTechId
		/// </summary>
		public string ReceiptTechId
		{
			get
			{
				if (this._drSupplyRow.IsNull(SUPPLY.ReceiptTechId))
					return string.Empty;

				return this._drSupplyRow[SUPPLY.ReceiptTechId].ToString();
			}
			set
			{
				this._drSupplyRow[SUPPLY.ReceiptTechId] = value;
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="33"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="34"> 
		///		<ExpectedInput>empty string</ExpectedInput>
		///		<ExpectedOutput>broken rule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// InvoiceNumber
		/// </summary>
		public string InvoiceNumber
		{
			get
			{
				if (this._drSupplyRow.IsNull(SUPPLY.InvoiceNumber))
					return string.Empty;

				return this._drSupplyRow[SUPPLY.InvoiceNumber].ToString();
			}
			set
			{
				if (value == string.Empty || value.Length > 20)
				{
					RuleBroken(MethodBase.GetCurrentMethod().Name, true);
					return;
				}

				RuleBroken(MethodBase.GetCurrentMethod().Name, false);
				this._drSupplyRow[SUPPLY.InvoiceNumber] = value;
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="35"> 
		///		<ExpectedInput>int</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="36"> 
		///		<ExpectedInput>1000</ExpectedInput>
		///		<ExpectedOutput>broken rule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LotQuantity
		/// </summary>
		public int LotQuantity
		{
			get
			{
				if (this._drSupplyRow.IsNull(SUPPLY.LotQuantity))
					return int.MinValue;

				return System.Convert.ToInt32(this._drSupplyRow[SUPPLY.LotQuantity]);
			}
			set
			{
				if (value == int.MinValue || value > 999)
				{
					RuleBroken(MethodBase.GetCurrentMethod().Name, true);
					return;
				}

				RuleBroken(MethodBase.GetCurrentMethod().Name, false);
				this._drSupplyRow[SUPPLY.LotQuantity] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2055"> 
		///		<ExpectedInput>Date</ExpectedInput>
		///		<ExpectedOutput>Date</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2056"> 
		///		<ExpectedInput>Future date</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// InventoryChangeDate
		/// </summary>
		public System.DateTime InventoryChangeDate
		{
			get
			{
				if (this._drSupplyRow.IsNull(SUPPLY.InventoryChangeDate))
					return System.DateTime.MinValue;

				return (System.DateTime)this._drSupplyRow[SUPPLY.InventoryChangeDate];
			}
			set
			{
				if (value > BOL.VBECSDateTime.GetDivisionCurrentDateTime())
				{
					RuleBroken(MethodBase.GetCurrentMethod().Name, true);
					return;
				}

				RuleBroken(MethodBase.GetCurrentMethod().Name, false);
				this._drSupplyRow[SUPPLY.InventoryChangeDate] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="37"> 
		///		<ExpectedInput>DateTime</ExpectedInput>
		///		<ExpectedOutput>DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="38"> 
		///		<ExpectedInput>MinDate</ExpectedInput>
		///		<ExpectedOutput>BrokenRule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LotExpirationDate
		/// </summary>
		public System.DateTime LotExpirationDate
		{
			get
			{
				if (this._drSupplyRow.IsNull(SUPPLY.LotExpirationDate))
					return System.DateTime.MinValue;

				return (System.DateTime)this._drSupplyRow[SUPPLY.LotExpirationDate];
			}
			set
			{
				if (value == System.DateTime.MinValue || value < this.ReceiptDate)
				{
					RuleBroken(MethodBase.GetCurrentMethod().Name, true);
					return;
				}

				RuleBroken(MethodBase.GetCurrentMethod().Name, false);
				this._drSupplyRow[SUPPLY.LotExpirationDate] = value;
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="39"> 
		///		<ExpectedInput>bool</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="40"> 
		///		<ExpectedInput>false</ExpectedInput>
		///		<ExpectedOutput>BrokenRule for Comments</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// SatisfactoryIndicator
		/// </summary>
		public bool SatisfactoryIndicator
		{
			get
			{
				if (this._drSupplyRow.IsNull(SUPPLY.SatisfactoryIndicator))
					return false;

				return (bool)this._drSupplyRow[SUPPLY.SatisfactoryIndicator];
			}
			set
			{
				//Need to unbreak the rule for logging new supplies
				RuleBroken(MethodBase.GetCurrentMethod().Name, false);
				this._drSupplyRow[SUPPLY.SatisfactoryIndicator] = value;

				//This breaks the comment required rule automatically when the 
				//user selects unsatisfactory -- unless they have something already there
				if (value == true)
					RuleBroken(SUPPLY.Comments, false);
				else
					RuleBroken(SUPPLY.Comments, (this.Comments.Length == 0));
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="41"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="42"> 
		///		<ExpectedInput>256 char string</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Comments
		/// </summary>
		public override string Comments
		{
			get
			{
				if (this._drSupplyRow.IsNull(SUPPLY.Comments))
					return string.Empty;

				return this._drSupplyRow[SUPPLY.Comments].ToString();
			}
			set
			{
				//If they sent in an empty string, and the supply is satisfactory, let it pass - for new supplies
				if (this.IsNew)
				{
					if ((value == string.Empty && !this.SatisfactoryIndicator) || value.Length > 255)
					{
						RuleBroken(MethodBase.GetCurrentMethod().Name, true);
						return;
					}
				}
				else
				{
					if ((value == string.Empty) || (value.Length > 255))
					{
						RuleBroken(MethodBase.GetCurrentMethod().Name, true);
						return;
					}
				}

				RuleBroken(MethodBase.GetCurrentMethod().Name, false);
				this._drSupplyRow[SUPPLY.Comments] = value;
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="43"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="44"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Read-Only, should be set during MakeNew() of class
		/// </summary>
		public Guid SupplyGuid
		{
			get
			{
				if (this._drSupplyRow.IsNull(SUPPLY.SupplyGuid))
				{
					return System.Guid.Empty;
				}
				return (System.Guid)this._drSupplyRow[SUPPLY.SupplyGuid];
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="45"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="46"> 
		///		<ExpectedInput>Empty String</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets lot number of supply
		/// </summary>
		public string LotNumber
		{
			get
			{
				if (this._drSupplyRow.IsNull(SUPPLY.LotNumber))
				{
					return string.Empty;
				}
				return this._drSupplyRow[SUPPLY.LotNumber].ToString();
			}
			set
			{
				if (value == string.Empty || value.Length > 50)
				{
					RuleBroken(MethodBase.GetCurrentMethod().Name, true);
					return;
				}

				RuleBroken(MethodBase.GetCurrentMethod().Name, false);
				this._drSupplyRow[SUPPLY.LotNumber] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="47"> 
		///		<ExpectedInput>int</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="48"> 
		///		<ExpectedInput>invalid id</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets supply type
		/// </summary>
		public int SupplyTypeId
		{
			get
			{
				if (this._drSupplyRow.IsNull(SUPPLY.SupplyTypeId))
					return 0;
				
				return System.Convert.ToInt16(this._drSupplyRow[SUPPLY.SupplyTypeId]);
			}
			set
			{
				//Since this corresponds to a table ID field, we check for 0
				if (value <= 0)
				{
					RuleBroken("SupplyType", true);
					return;
				}

				RuleBroken("SupplyType", false);
				this._drSupplyRow[SUPPLY.SupplyTypeId] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4393"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4394"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the supply type text
		/// </summary>
		/// 
		public string SupplyTypeText
		{
			get
			{
				if (this._drSupplyRow.IsNull(Common.VbecsTables.SupplyType.SupplyTypeText))
					return string.Empty;
				
				return this._drSupplyRow[Common.VbecsTables.SupplyType.SupplyTypeText].ToString();
			}
			//Really shouldn't have this set in here...but, it's fast and simple
			set
			{
				this._drSupplyRow[Common.VbecsTables.SupplyType.SupplyTypeText] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="49"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="50"> 
		///		<ExpectedInput>Empty String</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets supply manufacturer name
		/// </summary>
		public string ManufacturerName
		{
			get
			{
				if (this._drSupplyRow.IsNull(SUPPLY.ManufacturerName))
					return string.Empty;

				return this._drSupplyRow[SUPPLY.ManufacturerName].ToString();
			}
			set
			{
				if (value == string.Empty || value.Length > 50)
				{
					RuleBroken(MethodBase.GetCurrentMethod().Name, true);				
					return;
				}
			
				RuleBroken(MethodBase.GetCurrentMethod().Name, false);
				this._drSupplyRow[SUPPLY.ManufacturerName] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="51"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="52"> 
		///		<ExpectedInput>empty string</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// divisionCode - in case Supply is not division specific
		/// </summary>
		public string DivisionCode
		{
			get
			{
				if (this._drSupplyRow.IsNull(SUPPLY.DivisionCode))
					return Common.LogonUser.LogonUserDivisionCode;

				return this._drSupplyRow[SUPPLY.DivisionCode].ToString();
			}
			set
			{
				if (value == string.Empty || value.Length > 5)
				{
					throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument(MethodBase.GetCurrentMethod().Name).ResString);
				}
				this._drSupplyRow[SUPPLY.DivisionCode] = value;
			}
		}

		///<Developers>
		///	<Developer>Greg</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6503"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6504"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Exception that may have been raised during UC 
		/// </summary>
		public BOL.ExceptionReport LotException
		{
			get
			{
				return (this._lotException);
			}
			set
			{
				this._lotException = value;
			}
		}

		#endregion

		#region Methods

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="53"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="54"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public function to retrieve valid supply in inventory matching search parameters
		/// </summary>
		/// <returns>Supply table</returns>
		public System.Data.DataTable GetSupply()
		{
			return DAL.Supply.GetSupplies((Common.SupplyType)this.SupplyTypeId, this.ManufacturerName, this.LotNumber, false);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="55"> 
		///		<ExpectedInput>SupplyType, bool, Invoice</ExpectedInput>
		///		<ExpectedOutput>Arraylist</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="56"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Public static function to retrieve supply in inventory with no search guidelines
		/// </summary>
		/// <param name="supplyType">Limits the search</param>
		/// <param name="includeAll">Include all Expired/NonExpired Supplies in list</param>
		/// <param name="limitingInvoice">Invoice to restrict list too</param>
		/// <returns>SupplyGuid Arraylist</returns>
		public static System.Collections.ArrayList GetSupplyList(Common.SupplyType supplyType, bool includeAll, string limitingInvoice)
		{
			System.Collections.ArrayList supplyList = new System.Collections.ArrayList();
			System.Data.DataTable dt = DAL.Supply.GetSupplies(supplyType, string.Empty, string.Empty, includeAll);
			
			foreach(System.Data.DataRow dr in dt.Rows)
			{
				BOL.Supply tmpObj = new BOL.Supply(dr);

				if (limitingInvoice == string.Empty)
				{
					supplyList.Add(tmpObj);
				}
				else
				{
					if (tmpObj.InvoiceNumber == limitingInvoice)
					{
						supplyList.Add(tmpObj);
					}
				}
			}

			return(supplyList);
		}

		///<Developers>
		///	<Developer></Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7739"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7740"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets supplies (non-Wafer/Containers) - We don't want a Modification to get containers/waafers inside it's supply list
		/// </summary>
		/// <param name="includeAll"></param>
		/// <returns></returns>
		public static System.Collections.ArrayList GetSupplyListSansContainerSCD(bool includeAll)
		{
			System.Collections.ArrayList supplyList = new System.Collections.ArrayList();
			System.Data.DataTable dt = DAL.Supply.GetSupplies(Common.SupplyType.AllSupplies, string.Empty, string.Empty, includeAll);
			
			foreach(System.Data.DataRow dr in dt.Rows)
			{
				BOL.Supply tmpObj = new BOL.Supply(dr);

				if (tmpObj.SupplyTypeId == (int)Common.SupplyType.Containers ||
					tmpObj.SupplyTypeId == (int)Common.SupplyType.SCDWafer)
				{
					//Ignore it
				}
				else
				{
					supplyList.Add(tmpObj);
				}
			}

			return(supplyList);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2818"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Guid</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2819"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Empty Guid</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets the SupplyGuid of an existing supply (by invoice number, lot number, suppy type)
		/// </summary>
		/// <returns>EmptyGuid if not found</returns>
		public System.Guid GetSupplyByInvoiceLotNumber()
		{
			System.Guid supplyGuid = System.Guid.Empty;
			System.Data.DataTable dtSupply = DAL.Supply.GetSupplyByInvoiceLotNumber(this.InvoiceNumber, this.LotNumber, this.SupplyTypeId);

			if (dtSupply.Rows.Count > 0)
				supplyGuid = (System.Guid)dtSupply.Rows[0][SUPPLY.SupplyGuid];

			return(supplyGuid);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2820"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Arraylist</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2821"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetSupplyDetails - retuns hashtable of types, arraylist of manufacturers in arraylist
		/// </summary>
		/// <returns></returns>
		public static System.Collections.ArrayList GetSupplyDetails()
		{
			System.Collections.Hashtable supplyTypeHash = new System.Collections.Hashtable();
			System.Collections.ArrayList supplyManufacturerList = new System.Collections.ArrayList();
			System.Collections.ArrayList supplyData = new System.Collections.ArrayList();

			System.Data.DataTable dtManufacturers = DAL.Supply.GetManufacturerNameList();
			System.Data.DataTable dtSupplyTypes = DAL.Supply.GetSupplyTypeList();
			
			foreach(System.Data.DataRow dr in dtSupplyTypes.Rows)
			{
				supplyTypeHash.Add((int)dr[SUPPLYTYPE.SupplyTypeId], dr[SUPPLYTYPE.SupplyTypeText].ToString());
			}

			foreach(System.Data.DataRow dr in dtManufacturers.Rows)
			{
				supplyManufacturerList.Add(dr[SUPPLY.ManufacturerName].ToString());
			}

			supplyData.Add(supplyTypeHash);
			supplyData.Add(supplyManufacturerList);

			return(supplyData);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		/// <summary>
		/// Loads the local datarow object from this datarow parameter
		/// </summary>
		/// <param name="dataRow"></param>
		protected override void LoadFromDataRow(System.Data.DataRow dataRow)
		{
			this._drSupplyRow = this._dtSupply.NewRow();

			if (dataRow.Table.Columns.Contains(SUPPLY.Comments))
			{
				this._drSupplyRow[SUPPLY.Comments] = dataRow[SUPPLY.Comments];
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.DivisionCode))
			{
				this._drSupplyRow[SUPPLY.DivisionCode] = dataRow[SUPPLY.DivisionCode];
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.SupplyGuid))
			{
				this._drSupplyRow[SUPPLY.SupplyGuid] = dataRow[SUPPLY.SupplyGuid];
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.SupplyName))
			{
				this._drSupplyRow[SUPPLY.SupplyName] = dataRow[SUPPLY.SupplyName];
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.SupplyTypeId))
			{
				this._drSupplyRow[SUPPLY.SupplyTypeId] = dataRow[SUPPLY.SupplyTypeId];
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.SatisfactoryIndicator))
			{
				this._drSupplyRow[SUPPLY.SatisfactoryIndicator] = dataRow[SUPPLY.SatisfactoryIndicator];
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.AvailableQuantity))
			{
				this._drSupplyRow[SUPPLY.AvailableQuantity] = dataRow[SUPPLY.AvailableQuantity];
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.InvoiceNumber))
			{
				this._drSupplyRow[SUPPLY.InvoiceNumber] = dataRow[SUPPLY.InvoiceNumber];
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.LotNumber))
			{
				this._drSupplyRow[SUPPLY.LotNumber] = dataRow[SUPPLY.LotNumber];
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.LotExpirationDate))
			{
				this._drSupplyRow[SUPPLY.LotExpirationDate] = dataRow[SUPPLY.LotExpirationDate];
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.ManufacturerName))
			{
				this._drSupplyRow[SUPPLY.ManufacturerName] = dataRow[SUPPLY.ManufacturerName];
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.ReceiptDate))
			{
				this._drSupplyRow[SUPPLY.ReceiptDate] = dataRow[SUPPLY.ReceiptDate];
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.ReceiptTechId))
			{
				this._drSupplyRow[SUPPLY.ReceiptTechId] = dataRow[SUPPLY.ReceiptTechId];
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.LotQuantity))
			{
				this._drSupplyRow[SUPPLY.LotQuantity] = dataRow[SUPPLY.LotQuantity];
			}

			if (dataRow.Table.Columns.Contains(SUPPLYTYPE.SupplyTypeText))
			{
				this._drSupplyRow[SUPPLYTYPE.SupplyTypeText] = dataRow[SUPPLYTYPE.SupplyTypeText];
			}

			//Added 9/22/2004 - new field for update supplies GUI
			if (dataRow.Table.Columns.Contains(SUPPLY.InventoryChangeDate))
			{
				this._drSupplyRow[SUPPLY.InventoryChangeDate] = dataRow[SUPPLY.InventoryChangeDate];
			}

			//Added on 8/26/2004 - for insert/update
			if (dataRow.Table.Columns.Contains(SUPPLYTYPE.LastUpdateDate))
			{
				this.LastUpdateDate = (System.DateTime)dataRow[SUPPLYTYPE.LastUpdateDate];
			}

			if (dataRow.Table.Columns.Contains(SUPPLYTYPE.LastUpdateUser))
			{
				this.LastUpdateUser = dataRow[SUPPLYTYPE.LastUpdateUser].ToString();
			}

			if (dataRow.Table.Columns.Contains(SUPPLYTYPE.RowVersion))
			{
				this.RowVersion = (byte[])dataRow[SUPPLYTYPE.RowVersion];
			}
			//End 8/26/2004 changes

			this.IsDirty = true;

		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="57"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="58"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads a datarow with the data from the object
		/// </summary>
		/// <param name="dataRow"></param>
		/// <returns></returns>
		public override System.Data.DataRow LoadDataRowFromThis(System.Data.DataRow dataRow)
		{
			if (dataRow.Table.Columns.Contains(SUPPLY.Comments))
			{
				if (!_drSupplyRow.IsNull(SUPPLY.Comments))
					dataRow[SUPPLY.Comments] = this.Comments;
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.DivisionCode))
			{
				//DivisionCode will default if not provided
				dataRow[SUPPLY.DivisionCode] = this.DivisionCode;
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.SupplyGuid))
			{
				if (!_drSupplyRow.IsNull(SUPPLY.SupplyGuid))
					dataRow[SUPPLY.SupplyGuid] = this.SupplyGuid;
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.SupplyName))
			{
				if (!_drSupplyRow.IsNull(SUPPLY.SupplyName))
					dataRow[SUPPLY.SupplyName] = this.SupplyName;
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.SupplyTypeId))
			{
				if (!_drSupplyRow.IsNull(SUPPLY.SupplyTypeId))
					dataRow[SUPPLY.SupplyTypeId] = this.SupplyTypeId;
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.SatisfactoryIndicator))
			{
				if (!_drSupplyRow.IsNull(SUPPLY.SatisfactoryIndicator))
					dataRow[SUPPLY.SatisfactoryIndicator] = this.SatisfactoryIndicator;
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.AvailableQuantity))
			{
				if (!_drSupplyRow.IsNull(SUPPLY.AvailableQuantity))
					dataRow[SUPPLY.AvailableQuantity] = this.AvailableQuantity;
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.InvoiceNumber))
			{
				if (!_drSupplyRow.IsNull(SUPPLY.InvoiceNumber))
					dataRow[SUPPLY.InvoiceNumber] = this.InvoiceNumber;
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.LotQuantity))
			{
				if (!_drSupplyRow.IsNull(SUPPLY.LotQuantity))
					dataRow[SUPPLY.LotQuantity] = this.LotQuantity;
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.LotExpirationDate))
			{
				if (!_drSupplyRow.IsNull(SUPPLY.LotExpirationDate))
					dataRow[SUPPLY.LotExpirationDate] = this.LotExpirationDate;
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.LotNumber))
			{
				if (!_drSupplyRow.IsNull(SUPPLY.LotNumber))
					dataRow[SUPPLY.LotNumber] = this.LotNumber;
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.ManufacturerName))
			{
				if (!_drSupplyRow.IsNull(SUPPLY.ManufacturerName))
					dataRow[SUPPLY.ManufacturerName] = this.ManufacturerName;
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.ReceiptDate))
			{
				if (!_drSupplyRow.IsNull(SUPPLY.ReceiptDate))
					dataRow[SUPPLY.ReceiptDate] = this.ReceiptDate;
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.ReceiptTechId))
			{
				if (!_drSupplyRow.IsNull(SUPPLY.ReceiptTechId))
					dataRow[SUPPLY.ReceiptTechId] = this.ReceiptTechId;
			}

			//9/22/2004 - New field for updates - not inserts
			if (dataRow.Table.Columns.Contains(SUPPLY.InventoryChangeDate))
			{
				if (!_drSupplyRow.IsNull(SUPPLY.InventoryChangeDate))
					dataRow[SUPPLY.InventoryChangeDate] = this.InventoryChangeDate;
			}

			//Added 8/26/2004 - for insert/updates
			if (dataRow.Table.Columns.Contains(SUPPLY.LastUpdateUser))
			{
				dataRow[SUPPLY.LastUpdateUser] = this.LastUpdateUser;
			}

			if (dataRow.Table.Columns.Contains(SUPPLY.RowVersion))
			{
				//This blows
				if (this.RowVersion != null)
					dataRow[SUPPLY.RowVersion] = this.RowVersion;
			}
			//End 8/26/2004 changes

			return dataRow;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="21"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="22"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates a blank Supply DataTable
		/// </summary>
		/// <returns></returns>
		public static System.Data.DataTable GetEmptySupplyTable()
		{
			return DAL.Supply.GetEmptySupplyTable();
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1733"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>ArrayList</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1736"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates and arraylist of unique ManufacturerNames from the Supply Table
		/// </summary>
		/// <returns></returns>
		public static System.Collections.ArrayList GetManufacturerNameList()
		{
			System.Data.DataTable dtData = DAL.Supply.GetManufacturerNameList();

			System.Collections.ArrayList alNames = new System.Collections.ArrayList();

			foreach(System.Data.DataRow dr in dtData.Rows)
			{
				alNames.Add(dr[SUPPLY.ManufacturerName].ToString());
			}

			return(alNames);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1739"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Array List of all previously entered supplies</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1740"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates and arraylist of unique SupplyName from the Supply Table
		/// </summary>
		/// <returns></returns>
		public static System.Collections.ArrayList GetSupplyNameList()
		{
			return GetSupplyNameList("");
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/23/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7178"> 
		///		<ExpectedInput>Valid manufacturer name</ExpectedInput>
		///		<ExpectedOutput>Array List of all previously entered supplies for manufacturer</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7179"> 
		///		<ExpectedInput>Invalid manufacturer name</ExpectedInput>
		///		<ExpectedOutput>Empty array list</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Creates and arraylist of unique SupplyName from the Supply Table for a given manufacturer
		/// </summary>
		/// <param name="manufacturerName"></param>
		/// <returns></returns>
		public static System.Collections.ArrayList GetSupplyNameList(string manufacturerName)
		{
			System.Data.DataTable dtData = DAL.Supply.GetSupplyNameList(manufacturerName);

			System.Collections.ArrayList alNames = new System.Collections.ArrayList();

			foreach(System.Data.DataRow dr in dtData.Rows)
			{
				alNames.Add(dr[SUPPLY.SupplyName].ToString());
			}

			return(alNames);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1743"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Arralist[]</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1744"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ArrayList[0] = SupplyTypeIds, ArrayList[1] = SupplyTypeText
		/// </summary>
		/// <returns></returns>
		public static System.Collections.ArrayList[] GetSupplyTypeList()
		{
			System.Data.DataTable dtData = DAL.Supply.GetSupplyTypeList();

			System.Collections.ArrayList supplyTypeIds = new System.Collections.ArrayList();
			System.Collections.ArrayList supplyTypeText = new System.Collections.ArrayList();

			foreach(System.Data.DataRow dr in dtData.Rows)
			{
				supplyTypeIds.Add((int)dr[SUPPLYTYPE.SupplyTypeId]);
				supplyTypeText.Add(dr[SUPPLYTYPE.SupplyTypeText].ToString());
			}

			//This used to return a hashtable, but it seemed like loading/unloading the hashtable
			//messed up the order of things (and hence, the sort order) and so on...
			System.Collections.ArrayList[] supplyData = {supplyTypeIds, supplyTypeText};
			return(supplyData);
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1755"> 
		///		<ExpectedInput>ArrayList of Supply objects</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1756"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Parses through the array list of Supply objects and creates a datatable for DAL
		/// </summary>
		/// <returns></returns>
		public static bool InsertSupplies(System.Collections.ArrayList alSupplies, Common.UpdateFunction lastUpdateFunctionId)
		{
			System.Data.DataTable dtInsert = DAL.Supply.GetInsertSupplyDataTable();
			System.Guid [] supplyGuids = new Guid[alSupplies.Count];
			int nCtr = 0;
			for (int ndx=0; ndx < alSupplies.Count; ndx++)
			{
				BOL.Supply newSupply = (BOL.Supply)alSupplies[ndx];
				
				System.Data.DataRow drAdd = dtInsert.NewRow();
				drAdd = newSupply.LoadDataRowFromThis(drAdd);
				supplyGuids[nCtr] = (System.Guid) drAdd[Common.VbecsTables.Supply.SupplyGuid];
				drAdd[SUPPLY.LastUpdateUser] = Common.LogonUser.LogonUserName;

				//This might get removed if a dropdown is created on the GUI
				drAdd[SUPPLY.ReceiptTechId] = Common.LogonUser.LogonUserName;

				dtInsert.Rows.Add(drAdd);
				nCtr = nCtr + 1;
			}
			System.Data.DataTable dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.LoginSupply, supplyGuids, Common.WorkloadTransactionType.Miscellanious, false); 
			dtInsert = Common.Utility.AppendLastUpdateInformation(dtInsert, lastUpdateFunctionId);
			dtWorkloadEvents = Common.Utility.AppendLastUpdateInformation(dtWorkloadEvents, lastUpdateFunctionId);
			return(DAL.Supply.InsertSupplies(dtInsert, dtWorkloadEvents));

		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/25/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1820"> 
		///		<ExpectedInput>LastUpdateFunctionId</ExpectedInput>
		///		<ExpectedOutput>bool</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1821"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Saves the current supply object changes -- for UpdateSupplyUsage - UC81
		/// Could someday be expanded to use the spUpdateSupply sproc tha updates ALL fields
		/// </summary>
		/// <returns></returns>
		public bool Update(Common.UpdateFunction lastUpdateFunctionId)
		{
			System.Data.DataTable dtSupply = new System.Data.DataTable(SUPPLY.TableName);

			dtSupply.Columns.Add(SUPPLY.AvailableQuantity, typeof(int));
			dtSupply.Columns.Add(SUPPLY.Comments, typeof(string));
			dtSupply.Columns.Add(SUPPLY.DivisionCode, typeof(string));
			dtSupply.Columns.Add(SUPPLY.InventoryChangeDate, typeof(System.DateTime));
			dtSupply.Columns.Add(SUPPLY.RowVersion, typeof(byte[]));
			dtSupply.Columns.Add(SUPPLY.SupplyGuid, typeof(System.Guid));

			//Create our DataRow of update values
			System.Data.DataRow drUpdateSupply = dtSupply.NewRow();

			drUpdateSupply[SUPPLY.AvailableQuantity] = this.AvailableQuantity;
			drUpdateSupply[SUPPLY.Comments] = this.Comments;
			drUpdateSupply[SUPPLY.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			drUpdateSupply[SUPPLY.InventoryChangeDate] = this.InventoryChangeDate;
			drUpdateSupply[SUPPLY.RowVersion] = this.RowVersion;
			drUpdateSupply[SUPPLY.SupplyGuid] = this.SupplyGuid;

			dtSupply.Rows.Add(drUpdateSupply);

			//Get the Workload
			System.Guid[] workloadGuids = {this.SupplyGuid};
			System.Data.DataTable dtWorkload = BOL.WorkloadEvent.GenerateWorkloadData(Common.WorkloadProcessID.UpdateSupplyInventory, workloadGuids, Common.WorkloadTransactionType.Miscellanious, false);

			return (DAL.Supply.UpdateSupplyUsage(dtSupply, dtWorkload, lastUpdateFunctionId));
		}



		#endregion

	}
}


